/*
 * Decompiled with CFR 0.152.
 */
package com.autojav.core.license;

import com.autojav.core.ConfigManager;
import com.autojav.core.TerminalUtils;
import com.autojav.core.license.FeaturePermissions;
import com.autojav.core.license.License;
import com.autojav.core.license.LicenseValidator;
import com.autojav.core.license.VersionType;
import java.time.LocalDateTime;
import java.time.format.DateTimeFormatter;
import java.time.temporal.ChronoUnit;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;

public class LicenseManager {
    private static final Logger log = LoggerFactory.getLogger(LicenseManager.class);
    private static final String LICENSE_KEY_CONFIG = "license.key";
    private static final String TRIAL_USED_CONFIG = "license.trial.used";
    private static final String TRIAL_START_CONFIG = "license.trial.start";
    private License currentLicense;
    private final ConfigManager configManager = new ConfigManager();

    public LicenseManager() {
        this.loadLicense();
    }

    public void loadLicense() {
        try {
            String licenseKey = this.configManager.get(LICENSE_KEY_CONFIG);
            if (licenseKey != null && !licenseKey.isEmpty()) {
                LicenseValidator.ValidationResult result = LicenseValidator.validate(licenseKey);
                if (result.isValid()) {
                    this.currentLicense = result.getLicense();
                    log.info("\u5df2\u52a0\u8f7d\u6b63\u5f0f\u8bb8\u53ef\u8bc1: {}", (Object)this.currentLicense.getVersionType().getName());
                    return;
                }
                log.warn("\u8bb8\u53ef\u8bc1\u9a8c\u8bc1\u5931\u8d25: {}", (Object)result.getMessage());
            }
            if (this.loadTrialLicense()) {
                return;
            }
            this.currentLicense = this.createFreeLicense();
            log.info("\u672a\u627e\u5230\u6709\u6548\u8bb8\u53ef\u8bc1\uff0c\u4f7f\u7528\u514d\u8d39\u7248");
        }
        catch (Exception e) {
            log.error("\u52a0\u8f7d\u8bb8\u53ef\u8bc1\u5931\u8d25", e);
            this.currentLicense = this.createFreeLicense();
        }
    }

    private boolean loadTrialLicense() {
        String trialUsed = this.configManager.get(TRIAL_USED_CONFIG);
        if (!"true".equals(trialUsed)) {
            return this.startTrial();
        }
        String trialStartStr = this.configManager.get(TRIAL_START_CONFIG);
        if (trialStartStr != null) {
            try {
                LocalDateTime trialStart = LocalDateTime.parse(trialStartStr);
                LocalDateTime trialEnd = trialStart.plusDays(7L);
                if (LocalDateTime.now().isBefore(trialEnd)) {
                    this.currentLicense = this.createTrialLicense(trialStart, trialEnd);
                    long daysLeft = ChronoUnit.DAYS.between(LocalDateTime.now(), trialEnd);
                    log.info("\u8bd5\u7528\u8bb8\u53ef\u8bc1\u6709\u6548\uff0c\u5269\u4f59 {} \u5929", (Object)daysLeft);
                    return true;
                }
                log.info("\u8bd5\u7528\u671f\u5df2\u8fc7\u671f");
            }
            catch (Exception e) {
                log.warn("\u89e3\u6790\u8bd5\u7528\u65f6\u95f4\u5931\u8d25", e);
            }
        }
        return false;
    }

    private boolean startTrial() {
        try {
            LocalDateTime now = LocalDateTime.now();
            this.configManager.set(TRIAL_USED_CONFIG, "true");
            this.configManager.set(TRIAL_START_CONFIG, now.toString());
            this.configManager.save(false);
            LocalDateTime trialEnd = now.plusDays(7L);
            this.currentLicense = this.createTrialLicense(now, trialEnd);
            TerminalUtils.printSuccess("\ud83c\udf89 \u5df2\u5f00\u542f 7 \u5929\u56e2\u961f\u7248\u8bd5\u7528\uff01");
            log.info("\u8bd5\u7528\u8bb8\u53ef\u8bc1\u521b\u5efa\u6210\u529f");
            return true;
        }
        catch (Exception e) {
            log.error("\u521b\u5efa\u8bd5\u7528\u8bb8\u53ef\u8bc1\u5931\u8d25", e);
            return false;
        }
    }

    private License createFreeLicense() {
        License license = new License();
        license.setLicenseId("FREE-" + System.currentTimeMillis());
        license.setVersionType(VersionType.FREE);
        license.setStartTime(LocalDateTime.now());
        license.setExpireTime(LocalDateTime.now().plusYears(100L));
        license.setPermissions(new FeaturePermissions(VersionType.FREE));
        license.setStatus(License.LicenseStatus.ACTIVE);
        return license;
    }

    private License createTrialLicense(LocalDateTime start, LocalDateTime end) {
        License license = new License();
        license.setLicenseId("TRIAL-" + LicenseValidator.getMachineFingerprint());
        license.setVersionType(VersionType.TEAM);
        license.setStartTime(start);
        license.setExpireTime(end);
        license.setPermissions(new FeaturePermissions(VersionType.TEAM));
        license.setStatus(License.LicenseStatus.TRIAL);
        license.setTrial(true);
        return license;
    }

    public boolean validateLicense() {
        if (this.currentLicense == null) {
            return false;
        }
        return this.currentLicense.isValid() && LocalDateTime.now().isBefore(this.currentLicense.getExpireTime());
    }

    public boolean hasPermission(String feature) {
        if (this.currentLicense == null) {
            return false;
        }
        if (LocalDateTime.now().isAfter(this.currentLicense.getExpireTime())) {
            return FeaturePermissions.isFreeFeature(feature);
        }
        return this.currentLicense.hasPermission(feature);
    }

    public License getCurrentLicense() {
        return this.currentLicense;
    }

    public boolean activateLicense(String licenseKey) {
        try {
            LicenseValidator.ValidationResult result = LicenseValidator.validate(licenseKey);
            if (!result.isValid()) {
                TerminalUtils.printError("\u8bb8\u53ef\u8bc1\u9a8c\u8bc1\u5931\u8d25: " + result.getMessage());
                return false;
            }
            License license = result.getLicense();
            if (license.getVersionType() == VersionType.FREE && this.currentLicense.getVersionType() != VersionType.FREE) {
                TerminalUtils.printWarning("\u65e0\u6cd5\u6fc0\u6d3b\u514d\u8d39\u7248\u8bb8\u53ef\u8bc1\uff0c\u60a8\u5f53\u524d\u5df2\u6709\u66f4\u9ad8\u7ea7\u522b\u8bb8\u53ef");
                return false;
            }
            this.configManager.set(LICENSE_KEY_CONFIG, licenseKey);
            this.configManager.save(false);
            this.currentLicense = license;
            TerminalUtils.printSuccess("\u2705 \u8bb8\u53ef\u8bc1\u6fc0\u6d3b\u6210\u529f\uff01");
            TerminalUtils.printInfo("\u7248\u672c: " + license.getVersionType().getName());
            TerminalUtils.printInfo("\u6709\u6548\u671f\u81f3: " + license.getExpireTime().format(DateTimeFormatter.ofPattern("yyyy-MM-dd")));
            log.info("\u8bb8\u53ef\u8bc1\u6fc0\u6d3b\u6210\u529f: {}", (Object)license.getVersionType().getName());
            return true;
        }
        catch (Exception e) {
            log.error("\u6fc0\u6d3b\u8bb8\u53ef\u8bc1\u5931\u8d25", e);
            TerminalUtils.printError("\u6fc0\u6d3b\u5931\u8d25: " + e.getMessage());
            return false;
        }
    }

    public void showLicenseInfo() {
        if (this.currentLicense == null) {
            TerminalUtils.printError("\u672a\u627e\u5230\u8bb8\u53ef\u8bc1\u4fe1\u606f");
            return;
        }
        TerminalUtils.printInfo("\u2550\u2550\u2550\u2550\u2550\u2550\u2550\u2550\u2550\u2550\u2550\u2550\u2550\u2550\u2550\u2550\u2550\u2550\u2550\u2550\u2550\u2550\u2550\u2550\u2550\u2550\u2550\u2550\u2550\u2550\u2550\u2550\u2550");
        TerminalUtils.printInfo("\ud83d\udccb \u8bb8\u53ef\u8bc1\u4fe1\u606f");
        TerminalUtils.printInfo("\u2550\u2550\u2550\u2550\u2550\u2550\u2550\u2550\u2550\u2550\u2550\u2550\u2550\u2550\u2550\u2550\u2550\u2550\u2550\u2550\u2550\u2550\u2550\u2550\u2550\u2550\u2550\u2550\u2550\u2550\u2550\u2550\u2550");
        Object versionStr = this.currentLicense.getVersionType().getName();
        if (this.currentLicense.isTrial()) {
            versionStr = (String)versionStr + " (\u8bd5\u7528)";
        }
        TerminalUtils.printInfo("\u7248\u672c\u7c7b\u578b: " + (String)versionStr);
        Object statusStr = this.currentLicense.getStatus().getName();
        if (!this.validateLicense()) {
            statusStr = (String)statusStr + " (\u5df2\u8fc7\u671f)";
        }
        TerminalUtils.printInfo("\u72b6\u6001: " + (String)statusStr);
        TerminalUtils.printInfo("\u6709\u6548\u671f: " + this.currentLicense.getStartTime().format(DateTimeFormatter.ofPattern("yyyy-MM-dd")) + " \u81f3 " + this.currentLicense.getExpireTime().format(DateTimeFormatter.ofPattern("yyyy-MM-dd")));
        long daysLeft = ChronoUnit.DAYS.between(LocalDateTime.now(), this.currentLicense.getExpireTime());
        if (daysLeft < 0L) {
            TerminalUtils.printWarning("\u26a0\ufe0f  \u8bb8\u53ef\u8bc1\u5df2\u8fc7\u671f " + Math.abs(daysLeft) + " \u5929");
        } else if (daysLeft < 7L) {
            TerminalUtils.printWarning("\u26a0\ufe0f  \u8bb8\u53ef\u8bc1\u5373\u5c06\u8fc7\u671f\uff0c\u5269\u4f59 " + daysLeft + " \u5929");
        } else {
            TerminalUtils.printInfo("\u5269\u4f59\u5929\u6570: " + daysLeft + " \u5929");
        }
        TerminalUtils.printInfo("\u8bbe\u5907\u6307\u7eb9: " + LicenseValidator.getMachineFingerprint());
        TerminalUtils.printInfo("\u2550\u2550\u2550\u2550\u2550\u2550\u2550\u2550\u2550\u2550\u2550\u2550\u2550\u2550\u2550\u2550\u2550\u2550\u2550\u2550\u2550\u2550\u2550\u2550\u2550\u2550\u2550\u2550\u2550\u2550\u2550\u2550\u2550");
        TerminalUtils.printInfo("\ud83d\udd10 \u529f\u80fd\u6743\u9650");
        TerminalUtils.printInfo("\u2550\u2550\u2550\u2550\u2550\u2550\u2550\u2550\u2550\u2550\u2550\u2550\u2550\u2550\u2550\u2550\u2550\u2550\u2550\u2550\u2550\u2550\u2550\u2550\u2550\u2550\u2550\u2550\u2550\u2550\u2550\u2550\u2550");
        this.currentLicense.getPermissions().getAllPermissions().forEach((feature, allowed) -> {
            String featureName = this.translateFeatureName((String)feature);
            if (allowed.booleanValue()) {
                TerminalUtils.printSuccess("  \u2713 " + featureName);
            } else {
                TerminalUtils.printWarning("  \u2717 " + featureName + " (\u9700\u5347\u7ea7)");
            }
        });
        if (this.currentLicense.getVersionType() == VersionType.FREE || this.currentLicense.isTrial() || daysLeft < 7L) {
            TerminalUtils.printInfo("\u2550\u2550\u2550\u2550\u2550\u2550\u2550\u2550\u2550\u2550\u2550\u2550\u2550\u2550\u2550\u2550\u2550\u2550\u2550\u2550\u2550\u2550\u2550\u2550\u2550\u2550\u2550\u2550\u2550\u2550\u2550\u2550\u2550");
            TerminalUtils.printInfo("\ud83d\udca1 \u5347\u7ea7\u5230\u56e2\u961f\u7248\u89e3\u9501\u5168\u90e8\u529f\u80fd\uff1a");
            TerminalUtils.printInfo("   https://autojav.vercel.app");
        }
    }

    public boolean checkAndApplyRestriction(String feature) {
        if (this.hasPermission(feature)) {
            return true;
        }
        String featureName = this.translateFeatureName(feature);
        TerminalUtils.printWarning("\u26d4 " + featureName + " \u9700\u8981 " + this.getRequiredVersion(feature) + " \u53ca\u4ee5\u4e0a\u7248\u672c");
        TerminalUtils.printInfo("\u5f53\u524d\u7248\u672c: " + this.currentLicense.getVersionType().getName());
        if (this.currentLicense.getVersionType() == VersionType.FREE) {
            TerminalUtils.printInfo("");
            TerminalUtils.printInfo("\ud83d\udca1 \u60a8\u53ef\u4ee5\uff1a");
            TerminalUtils.printInfo("   1. \u5f00\u59cb 7 \u5929\u514d\u8d39\u8bd5\u7528\uff1aautojav license trial");
            TerminalUtils.printInfo("   2. \u8d2d\u4e70\u6b63\u5f0f\u8bb8\u53ef\u8bc1\uff1ahttps://autojav.vercel.app");
        } else {
            TerminalUtils.printInfo("");
            TerminalUtils.printInfo("\ud83d\udca1 \u8bf7\u8bbf\u95ee\u5b98\u7f51\u5347\u7ea7\uff1ahttps://autojav.vercel.app");
        }
        return false;
    }

    public boolean startTrialFromCommand() {
        String trialUsed = this.configManager.get(TRIAL_USED_CONFIG);
        if ("true".equals(trialUsed)) {
            TerminalUtils.printError("\u60a8\u5df2\u7ecf\u4f7f\u7528\u8fc7\u8bd5\u7528\u671f");
            return false;
        }
        if (this.currentLicense.getVersionType() != VersionType.FREE) {
            TerminalUtils.printWarning("\u60a8\u5df2\u6709\u6b63\u5f0f\u8bb8\u53ef\u8bc1\uff0c\u65e0\u9700\u8bd5\u7528");
            return false;
        }
        return this.startTrial();
    }

    private String getRequiredVersion(String feature) {
        switch (feature) {
            case "code.fix": 
            case "ai.audit": 
            case "team.collab": {
                return "\u56e2\u961f\u7248";
            }
            case "custom.template": 
            case "private.deploy": 
            case "ci.cd.integration": {
                return "\u4f01\u4e1a\u7248";
            }
        }
        return "\u514d\u8d39\u7248";
    }

    private String translateFeatureName(String feature) {
        switch (feature) {
            case "code.audit": {
                return "\u57fa\u7840\u4ee3\u7801\u5ba1\u8ba1";
            }
            case "code.fix": {
                return "AI \u4ee3\u7801\u4fee\u590d";
            }
            case "ai.audit": {
                return "AI \u6df1\u5ea6\u5ba1\u8ba1";
            }
            case "doc.generate": {
                return "\u63a5\u53e3\u6587\u6863\u751f\u6210";
            }
            case "team.collab": {
                return "\u56e2\u961f\u534f\u4f5c";
            }
            case "custom.template": {
                return "\u81ea\u5b9a\u4e49\u6a21\u677f";
            }
            case "private.deploy": {
                return "\u79c1\u6709\u5316\u90e8\u7f72";
            }
            case "ci.cd.integration": {
                return "CI/CD \u96c6\u6210";
            }
        }
        return feature;
    }
}

